//*********************************************************************************
//**
//** Project.........: Magnetic Loop Controller
//**
//**
//**
//** Platform........: AT90usb1286 @ 16MHz
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//** Initial version.: 2012-10-20, Loftur Jonasson, TF3LJ / VE2LJX
//**
//**
#define                VERSION "0.06"   // 2013-04-21
//**
//** History.........: Check the ML.c file
//**
//*********************************************************************************

#ifndef _TF3LJ_ML_H_
#define _TF3LJ_ML_H_ 1

#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <avr/io.h>
#include <avr/wdt.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <util/delay.h>
#include <avr/power.h>
#include <lcd.h>	


//-----------------------------------------------------------------------------
// Features Selection
//-----------------------------------------------------------------------------  
// EEPROM settings Serial Number. Increment this number when firmware mods necessitate
// fresh "Factory Default Settings" to be forced into the EEPROM at first boot after
// an upgrade
#define COLDSTART_REF		0x03// When started, the firmware examines this "Serial Number
								// and enforce factory reset if there is a mismatch.
								// This is useful if the EEPROM structure has been modified
//-----------------------------------------------------------------------------
// LED Blink
//
// None, or only one of the four should be selected
#define FAST_LOOP_THRU_LED	0	// Blink PB2 LED every time, when going through the mainloop *OR*
#define	MS_LOOP_THRU_LED	0	// Blink PB2 LED every 1ms, when going through the mainloop *OR*
#define	MED_LOOP_THRU_LED	0	// Blink PB2 LED every 10ms, when going through the mainloop *OR*
#define SLOW_LOOP_THRU_LED	1	// Blink PB2 LED every 100ms, when going through the mainloop


//-----------------------------------------------------------------------------
//DEFS for LED
#define LED_PORT			PORTD		// port for the LED
#define LED_DDR				DDRD		// port for the LED
#define LED					(1<<6)		// pin for LED

//-----------------------------------------------------------------------------
//DEFS for various Magloop specific paramenters
#define NUM_PRESETS			30			// Number of preset memories

/*
//-----------------------------------------------------------------------------
//DEFS for LCD Display
// The below #defines have been moved from the lcd.h file
#define LCD_PORT			PORTC		// port for the LCD lines
#define LCD_DATA0_PORT		LCD_PORT	// port for 4bit data bit 0
#define LCD_DATA1_PORT		LCD_PORT	// port for 4bit data bit 1
#define LCD_DATA2_PORT		LCD_PORT	// port for 4bit data bit 2
#define LCD_DATA3_PORT		LCD_PORT	// port for 4bit data bit 3
#define LCD_DATA0_PIN		4			// pin  for 4bit data bit 0
#define LCD_DATA1_PIN		5			// pin  for 4bit data bit 1
#define LCD_DATA2_PIN		6			// pin  for 4bit data bit 2
#define LCD_DATA3_PIN		7			// pin  for 4bit data bit 3
#define LCD_RS_PORT			PORTC		// port for RS line
#define LCD_RS_PIN			0			// pin  for RS line
#define LCD_RW_PORT			PORTC		// port for RW line
#define LCD_RW_PIN			1			// pin  for RW line
#define LCD_E_PORT			PORTC		// port for Enable line
#define LCD_E_PIN			2			// pin  for Enable line
*/
/*
//-----------------------------------------------------------------------------
// A couple of #defines moved from lcd.h:
#define LCD_LINES			4			// number of visible lines of the display 
#define LCD_DISP_LENGTH		20			// visibles characters per line of the display
*/

//-----------------------------------------------------------------------------
// DEFS for the Rotary Encoder VFO function
// Configuration of the two input pins, Phase A and Phase B
// They can be set up to use any pin on two separate input ports
#define ENC_A_PORT		PORTD				// PhaseA port register
#define ENC_A_DDR		DDRD				// PhaseA port direction register
#define ENC_A_PORTIN	PIND				// PhaseA port input register
#define ENC_A_PIN		(1 << 7)			// PhaseA port pin
#define ENC_B_PORT		PORTD				// PhaseB port register
#define ENC_B_DDR		DDRD				// PhaseB port direction register
#define ENC_B_PORTIN	PIND				// PhaseB port input register
#define ENC_B_PIN		(1 << 5)			// PhaseB port pin

#define	ENC_RES_DIVIDE		1				// Default reduction of the Encoder Resolution
#define ENCODER_DIR_REVERSE	0	// Reverse the direction of the Rotary Encoder


//-----------------------------------------------------------------------------
// Definitions for the USART (Serial Port)
#define USART_DDR		DDRD
#define	USART_RXD		(1 << 2)			// PD2
#define USART_TXD		(1 << 3)			// PD3
#define BAUD 			103					// 9600 b/s at 16MHz clock


//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// Definitions for the Pushbutton Encoder functionality
#define ENC_PUSHB_PORT		PORTD
#define ENC_PUSHB_DDR		DDRD
#define	ENC_PUSHB_INPORT	PIND
#define	ENC_PUSHB_PIN		(1 << 4)		// PD4

#define ENC_PUSHB_MIN		1				// Min pushdown for valid push (x 10ms)
#define	ENC_PUSHB_MAX		10				// Min pushdown for memory save (x 10ms)

//-----------------------------------------------------------------------------
// Definitions for the Stepper Motor
#define STEP_PORT		PORTB
#define STEP_DDR		DDRB

//-----------------------------------------------------------------------------
// Bargraph bounds
#define BAR_MIN_VALUE		800	// -80dB
#define BAR_FULL_RANGE		1000// 100dB full range
#define	BAR_FINE_RES		160	// 16dB for full scale variation if in fine resolution mode
#define BAR_LENGTH			16	// 16 characters long


//-----------------------------------------------------------------------------
// Select Bargraph display style
#define BARGRAPH_STYLE_1	1	// Used if LCD bargraph alternatives.  N8LP LP-100 look alike bargraph         *OR*
#define BARGRAPH_STYLE_2	0	// Used if LCD bargraph alternatives.  Bargraph with level indicators          *OR*
#define BARGRAPH_STYLE_3	0	// Used if LCD bargraph alternatives.  Another bargraph with level indicators  *OR*
#define BARGRAPH_STYLE_4	0	// Used if LCD bargraph alternatives.  Original bargraph, Empty space enframed *OR*
#define BARGRAPH_STYLE_5	0	// Used if LCD bargraph alternatives.  True bargraph, Empty space is empty


//
//-----------------------------------------------------------------------------
// Miscellaneous software defines, functions and variables
//-----------------------------------------------------------------------------
//

//-----------------------------------------------------------------------------
// Flags

// DEFS for all kinds of Flags
extern uint8_t			Status;
#define ENC_CHANGE		(1 << 0)			// Indicates that Encoder value has been modified
#define SHORT_PUSH		(1 << 1)			// Short Push Button Action
#define	LONG_PUSH		(1 << 2)			// Long Push Button Action
#define FRQ_TIMER		(1 << 3)			// FRQ_TIMER and FRQ_STORE are used to update EEPROM if
#define FRQ_STORE		(1 << 4)			//  frequency is stable for a while after a recent change.
#define FRQ_RADIO		(1 << 5)			// CI-V frequency information received from Radio
//#define MODE_DISPLAY	(1 << 5)			// Display mode has changed

// Operation Mode Flags
extern	uint8_t			Menu_Mode;			// Which Menu Mode is active
//#define POWER_DB		(1 << 0)
//#define	POWER_W			(1 << 1)
//#define VOLTAGE			(1 << 2)
//#define BARGRAPH_FULL	(1 << 3)
//#define BARGRAPH_16dB	(1 << 4)
//#define MIXED			(1 << 5)
#define	CONFIG			(1 << 6)
//
#define	DEFAULT_MODE 	(1 << 0)			// Default Menu Mode

// Bool stuff
//#define	TRUE		1
//#define FALSE		0


//-----------------------------------------------------------------------------
// Structures and Unions

typedef struct 
{
		uint8_t		EEPROM_init_check;		// If value mismatch,
											// then update EEPROM to factory settings
		int32_t	presetFrq[NUM_PRESETS];		// Up to NUM_PRESETS (30) frequency calibration settings
		int32_t	presetPos[NUM_PRESETS];		// Up to NUM_PRESETS (30)  corresponding Stepper positions
											// (positions 2 - 20 are relative to first pos)
		int32_t	curFrq;						// Latest tuned frequency	
		int32_t	curPos;						// Latest stepper position	
} var_t;


//-----------------------------------------------------------------------------
// Global variables
extern	EEMEM 		var_t E;				// Default Variables in EEPROM
extern				var_t R;				// Runtime Variables in Ram

extern	int32_t		delta_Pos;				// Stepper position offset
extern	int32_t		stepper_track;			// Stepper position tracking

extern	int16_t		encOutput;				// Output From Encoder

extern	uint8_t		civ_read[];				// CI-V data as a 5 char bcd
//extern	uint32_t	civ_value;				// CI-V data as a 32bit integer

extern	uint8_t		Menu_disp_timer;		// Used for a timed display when returning from Menu

//-----------------------------------------------------------------------------
// Prototypes for functions
//

// ML_usart.c
extern void 		usart_Init(void);				// Initialize USART
extern uint8_t		usart_Receive(void);			// Return received char

// ML_Encoder.c
extern void 		encoder_Init(void);				// Initialise the Rotary Encoder
extern void			encoder_Scan(void);				// Scan the Rotary Encoder

// ML_Stepper.c
extern void 		stepper_Init(void);				// Initialise Stepper Outputs
extern void			stepper_Incr(void);				// Increment Stepper
extern void			stepper_Decr(void);				// Decrement Stepper
extern void			stepper_PwrOff(void);			// Power Off Stepper

// ML_CIVreceive.c
extern void			civ_Parser(void);				// Read and Parse Frequency from CIV interface

// Push Button and Rotary Encoder Menu functions
extern void			PushButtonMenu(void);

// ML_Display_Functions.c
extern void			display_stepper_pos(int32_t);
extern void			display_frq(int32_t);

// LCD Bargraph stuff
extern void 		lcdProgressBar(uint16_t progress, uint16_t maxprogress, uint8_t length);
extern void			lcd_bargraph_init(void);

#endif
